#ifndef _ATALK_LOGGER_H
#define _ATALK_LOGGER_H 1

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <limits.h>
#include <stdbool.h>
#include <stdio.h>

/* logger is used by pam modules */
#ifndef UAM_MODULE_EXPORT
#define UAM_MODULE_EXPORT
#endif

enum loglevels {
    log_none,
    log_severe,
    log_error,
    log_warning,
    log_note,
    log_info,
    log_debug,
    log_debug6,
    log_debug7,
    log_debug8,
    log_debug9,
    log_maxdebug
};

/*! this is the enum specifying all availiable logtypes */
enum logtypes {
    logtype_default,            /*!< default logtype, used when no other is specified */
    logtype_logger,             /*!< general logger */
    logtype_cnid,               /*!< catalog node id backends */
    logtype_afpd,               /*!< apple filing protocol daemon */
    logtype_dsi,                /*!< data stream interface layer */
    logtype_atalkd,             /*!< appletalk daemon */
    logtype_papd,               /*!< printer access protocol daemon */
    logtype_uams,               /*!< user authentication modules */
    logtype_fce,                /*!< filesystem change events */
    logtype_ad,                 /*!< appledouble */
    logtype_sl,                 /*!< spotlight */
    logtype_end_of_list_marker  /*!< don't put any logtypes after this */
};


/* Display Option flags
 * Redefine these so they don't interfere with syslog
 * These can be used in standard logging too */
#define logoption_nsrcinfo    0x04   /*!< don't log source info */
/* the following do not work anymore, they're only provided in order to not
 * break existing source code */
#define logoption_pid         0x01   /*!< log the pid with each message */
#define logoption_cons        0x02   /*!< log on the console if error logging */
#define logoption_ndelay      0x08   /*!< don't delay open */
#define logoption_perror      0x20   /*!< log to stderr as well */
#define logoption_nfile       0x40   /*!< ignore the file that called the log */
#define logoption_nline       0x80   /*!< ignore the line that called the log*/

/* Facility codes
 * Redefine these so they don't interfere with syslog */
#define logfacility_user        (1<<3)  /*!< random user-level messages */
#define logfacility_mail        (2<<3)  /*!< mail system */
#define logfacility_daemon      (3<<3)  /*!< system daemons */
#define logfacility_auth        (4<<3)  /*!< security/authorization messages */
#define logfacility_syslog      (5<<3)  /*!< messages generated by syslogd */
#define logfacility_lpr         (6<<3)  /*!< line printer subsystem */
#define logfacility_authpriv    (10<<3) /*!< security/auth messages (private) */
#define logfacility_ftp         (11<<3) /*!< ftp daemon */

/* =========================================================================
    Structure definitions
 */

/* Main log config */
typedef struct {
    bool           inited;        /*!< file log config initialized ? */
    bool           syslog_opened; /*!< syslog opened ? */
    bool           console;       /*!< if logging to console from a cli util */
    char           processname[16];
    int            syslog_facility;
    int            syslog_display_options;
} log_config_t;

/* This stores the config and options for one filelog type (e.g. logger, afpd etc.) */
typedef struct {
    bool           set;           /*!< set individually ? yes: changing default
			                       * doesnt change it. no: it changes it.*/
    bool           syslog;        /*!< This type logs to syslog */
    int            fd;            /*!< logfiles fd */
    enum loglevels level;         /*!< Log Level to put in this file */
    int            display_options;
    bool           timestamp_us;  /*!< Log time stamps in us instead of s */
} logtype_conf_t;


/* =========================================================================
    Global variables
 */

/* Make config accessible for LOG macro */
extern log_config_t log_config;

extern UAM_MODULE_EXPORT logtype_conf_t
type_configs[logtype_end_of_list_marker];

/* =========================================================================
    Global function decarations
 */

void setuplog(const char *loglevel, const char *logfile,
              const bool log_us_timestamp);
void set_processname(const char *processname);

/*! Setup the level and type of log that will be logged to syslog. */
void syslog_setup(int loglevel, enum logtypes logtype, int display_options,
                  int facility);

/*! LOG macro func no.1: log the message to file */
UAM_MODULE_EXPORT  void make_log_entry(enum loglevels loglevel,
                                       enum logtypes logtype, const char *file, const bool log_us_timestamp, int line,
                                       char *message, ...);

/*
 * How to write a LOG macro:
 * http://c-faq.com/cpp/debugmacs.html
 *
 * We choose the verbose form in favor of the obfuscated ones, its easier
 * to parse for human beings and facilitates expanding the macro for
 * inline checks for debug levels.
 */

#define LOG_MAX log_info

#ifdef NO_DEBUG

#define LOG(log_level, type, ...)                                       \
    do {                                                                \
        if (log_level <= LOG_MAX)                                       \
            if (log_level <= type_configs[type].level)                  \
                make_log_entry((log_level), (type), __FILE__, type_configs[type].timestamp_us, __LINE__,  __VA_ARGS__); \
    } while(0)

#else  /* ! NO_DEBUG */

#define LOG(log_level, type, ...)               \
    do {                                                                \
        if (log_level <= type_configs[type].level)                      \
            make_log_entry((log_level), (type), __FILE__, type_configs[type].timestamp_us, __LINE__,  __VA_ARGS__); \
    } while(0)

#endif  /* NO_DEBUG */

#endif /* _ATALK_LOGGER_H */
